      program RT
c
c***********************************************************************
c     Computing eigenvalues and eigenfunctions of the Rayleigh-Taylor  *
c     instability for incompressible plane plasma slab with constant   *
c     magnetic field and linear density profile using shooting method  *
c                                                                      *
c     sigma    - normalized density gradient                           *
c     tau      - normalized gravity parameter                          *
c     aknul2   - square of normalized horizontal wavenumber            *
c     theta    - angle between k_0 and B                               *
c     neiv     - number of eigenvalues to be computed (le.nmax)        *
c     int      - number of intervals for eigenfunctions (le.imax)      *
c     err      - relerr and abserr for subroutine ODE                  *
c     eps      - distance to stay away from continuum at eimax         *
c     iploteif - plot eigenfunction aksi(x,n), eta(x,n)                *
c                                                                      *
c     idiag    - diagnostic print switch in subroutines iter and shoot *
c***********************************************************************
c
      implicit double precision (a-h,o-z)
c
      parameter (nmax = 100)
      parameter (imax = 1000)
      common / comvar1 / theta, eimin, eimax, sigma, tau, aknul2
      common / comvar2 / neiv, int, idiag
      common / comgrid / x(0:imax)
      common / eival / oms(nmax), test(nmax)
      common / eifun / aksi(0:imax,nmax), eta(0:imax,nmax)
      dimension xx(2), yy(2), yyy(nmax)
c
      character*80 header, text, bottom
      character*10 date, time
c
      external sofode
      external P
      external Q
c
c***********************************************************************
      header = 'Program RT - version  3.4, 18/04/07'
c*********************************** ==== *****************************
c
      pi = 3.14159265358979d0
c
c     * input
      sigma  = 1.0
      tau    = 10.0
      aknul2 = 10.
      theta  = 0.5*pi
      neiv   = 10
      int    = 100
      err    = 1.e-8
      eps    = 1.e-3
      iploteif = 1
c
c     * diagnostics
      idiag = 0
c
c     * plotting range for eigenvalues
      omsmin = -4.
      omsmax = 3.
c
      write(6,'(/1x,a)') header
      call dati(date,time)
      write(6,'(8x,'' run  '', a8/14x,a8)') date, time
      write(text,'(''Rayleigh-Taylor, sigma ='',f5.1,
     &      '', tau ='',f5.1,'', aknul2 ='',f5.1,'', theta ='',f6.3)')
     &      sigma, tau, aknul2, theta
      write(6,*)
      write(6,*) text 
c
c     * initiate plotting
      call hgo
      call ppp
      call begplt('RT.ps')
      call lbltop(text,70)
      write(bottom,'(''RT'',a4)') header(22:25)
      call lblbot(bottom,6)
c
c     * check input for discrete modes
      cos2 = cos(theta)**2
      grav  = sigma*tau
      eimin = aknul2*cos2 - grav
      if(eimin.gt.0.) eimin = eimin/(1.+sigma)
      eimax = aknul2*cos2/(1.+sigma) - eps
      write(6,'(/'' eimin ='',f8.3,'', grav ='',f8.3, 
     &          '' eimax ='',f8.3)') eimin, grav, eimax
      if(grav.le.0.) then
         write(6,'('' exit: no eigenvalues below continuum'')')
         goto 1000
      elseif(eimin.ge.0.) then
         write(6,'('' only stable eigenvalues'')')
      else
         write(6,'('' full range of eigenvalues'')')
      endif
c
c     * check continuum range
      alfven0 = aknul2*cos2
      alfven1 = aknul2*cos2/(1.+sigma)
      write(6,'('' continuum from alfven1 ='', 1pe12.4,0p,
     &          '' to alfven0 = '',1pe12.4,0p)') alfven1, alfven0
c
c     * start computation discrete modes
c     * grid 
      x(0) = 0.
      do 10 i = 1,int
         x(i) = float(i)/float(int)
   10 continue
c
c     * check absence of zeros at eimin
      write(6,*)
      write(6,'('' At minimum:'')')
      call shoot(1,err,eimin,testmin,nz)
      write(6,'('' eimin ='',1pe12.4,0p,'', nz = '',i3,
     &          '', testmin ='',1pe12.4,0p)') eimin, nz, testmin
      a1 = eimin
      t1 = testmin
      if(nz.gt.0) then
         print 15
         goto 1000
      endif
c 
c     * determine maximum number of eigenvalues from eimax 
      write(6,*)
      write(6,'('' At maximum:'')') 
      call shoot(1,err,eimax,testmax,nz)
      write(6,'('' eimax ='',1pe12.4,0p,'', nz = '',i3,
     &          '', testmax ='',1pe12.4,0p)') eimax, nz, testmax
      if(nz.lt.neiv) neiv = nz 
c
c     * iteration
      write(6,'(/'' Start iteration, neiv ='',i2,'', int = '',i3/)')
     &              neiv,int
      do 100 n = 1,neiv
         ne = n
         if(ne.eq.1) then
            a = a1
            t = t1
         endif         
         call iter(ne,err,a,t,iquit)
         if(iquit.eq.1) goto 1000
         oms(n)  = a
         test(n) = t
         write(6,'('' eigenvalue: n ='', i2,'', oms ='',1pe12.4,0p,
     &             '', test ='', 1pe12.4,0p)') n, oms(n), test(n)
         if(iploteif.ne.0) call ploteif(ne,a)
         write(6,*)
c
  100 continue

c     * plot spectrum
c
  200 call nframe(7111,2111,1,omsmin,omsmax,-1.D+0,1.D+0,
     &            'spectrum',8,'oms',3,' ',0)
c
c     * plot continuous spectrum
      h = 0.008
      xx(1) = alfven1
      xx(2) = alfven0
      do 110 i = 0, 10
         hs = 0.1*(2*i-10)*h
         yy(1) = hs
         yy(2) = hs
      call lplot(1,1,1,xx,yy,-2,1,' ',0,' ',0,' ',0)
  110 continue
      xx(1) = alfven1
      xx(2) = alfven1
      yy(1) = -2.0*h  
      yy(2) =  2.0*h 
      call lplot(1,1,1,xx,yy,-2,1,' ',0,' ',0,' ',0)
      xx(1) = alfven0
      xx(2) = alfven0
      call lplot(1,1,1,xx,yy,-2,1,' ',0,' ',0,' ',0)
c
c     * plot discrete eigenvalues
      do 120 n = 1,neiv
         yyy(n) = 0.
  120 continue
      call lplot(1,1,2031,oms(1),yyy(1),-neiv,1,' ',0,' ',0,' ',0)
c
      write(6,'('' * plot spectrum''/)')
c
 1000 call finplt
      stop
c
    5 format(/1x,'***exit: invalid input for coms')
   15 format(/1x,'***exit: invalid result for eimin')
      end
c
      subroutine iter(ne,err,a,t,iquit)
c
c***********************************************************************
c     Iteration on eigenvalue parameter a (=oms) until the criterion   *
c     t (=aksi(int)) le 1.e-8 is satisfied                             *
c***********************************************************************
c
      implicit double precision (a-h,o-z)
c
      parameter (nmax = 100)
      common / comvar1 / theta, eimin, eimax, sigma, tau, aknul2
      common / comvar2 / neiv, int, idiag
      common / eival / oms(nmax), test(nmax)
c
      iquit = 0
c
c     * produce a1 such that nz = ne-1 (but not the eigenvalue ne-1)
      if(ne.eq.1) then 
         a1 = a
         t1 = t 
      else
         a1 = oms(ne-1)
         t1 = test(ne-1)
c        * step away from a zero
         step = err
         if(abs(t1).le.1.e-8) then
   10       step = 10.*step
            a1 = a1 + step
            call shoot(ne,err,a1,t1,nz)
            if(abs(t1).le.1.e-8) goto 10
         endif
      endif
      if(idiag.eq.1) write(6,'(''diag1, a1 ='',1pe12.4,0p,
     &                         '', t1 ='',1pe12.4,0p/)') a1,t1
c
c     * produce a2 such that nz = ne
      d  = eimax - a1
      a2 = a1
      sd = 1.
      do 20 na2 = 1,30
         d  = d/2.
         a2 = a2 + sd*d
         if(idiag.eq.1) write(6,'(''diag2, a2 ='',1pe12.4,0p)') a2
         call shoot(ne,err,a2,t2,nz)
         if(idiag.eq.1) write(6,'(''diag3, nz ='',i3, 
     &                            '', t2 ='',1pe12.4,0p/)') nz, t2
         if(nz.eq.ne) then
            if(abs(t2).le.1.e-8) goto 50
            goto 30
         elseif(nz.lt.ne) then
            a1 = a2
            t1 = t2
            sd = 1.
         elseif(nz.gt.ne) then
            sd = -1.
         endif
   20 continue
      goto 60
c
c     * now zoom in onto the eigenvalue a with nz = ne 
   30 do 40 na = 1,30
         a = a1 - (a2-a1)*t1/(t2-t1)
         if(idiag.eq.1) write(6,'(''diag4, a ='',1pe12.4,0p)') a
         call shoot(ne,err,a,t,nz)
         if(idiag.eq.1) write(6,'(''diag5, nz ='',i3, 
     &                            '', t ='',1pe12.4,0p)') nz, t
         if(abs(t).le.1.e-8) goto 50
         if(nz.eq.ne) then
            a2 = a
            t2 = t
         else
            a1 = a
            t1 = t
         endif
   40 continue
      goto 50
c
   50 continue
      return
c
   60 print 61
      iquit = 1
      return
   70 print 71
      iquit = 1
      return
c
   61 format(/1x,'***subroutine iter: no convergence for a2')
   71 format(/1x,'***subroutine iter: no convergence for a')
      end
c
      subroutine shoot(ne,err,a,t,nz)
c
c***********************************************************************
c     Shooting for eigenvalues and eigenfunctions by means of ODE      *
c***********************************************************************
c
      implicit double precision (a-h,o-z)
c
      parameter (nmax = 100)
      parameter (imax = 1000)
      common / comvar1 / theta, eimin, eimax, sigma, tau, aknul2
      common / comvar2 / neiv, int, idiag
      common / comgrid / x(0:imax)
      common / comact / actoms
      common / eifun / aksi(0:imax,nmax), eta(0:imax,nmax)
c
      dimension y(2), work(142)
      integer iwork(5)
c
      external sofode
      external P
      external Q
c
      actoms = a
      sin1 = sin(theta)
      x0   = 0.
      y(1) = 0.
      y(2) = 1.
      iflag = 1
      aksi(0,ne) = 0.
      eta(0,ne)  = -(sin1/aknul2)*y(2)/P(x(0))
      nz = 0
      ys = 1.
      do 10 i = 1,int
         call ODE(sofode,2,y,x0,x(i),err,err,iflag,work,iwork)
         if(iflag.ne.2) print 5,iflag,i
         If(y(1)*ys.le.0.) then
            nz = nz + 1
            ys = y(1)
         endif
         aksi(i,ne) = y(1)
         eta(i,ne)  = -(sin1/aknul2)*y(2)/P(x(i))
   10 continue
      t = y(1)
      return
c
    5 format(1x,'***subroutine shoot: calling ODE gives iflag =',i2,
     &          ' at i =', i3)
      end
c
      subroutine sofode(x,y,yp)
c
c***********************************************************************
c     System of first order differential equations                     *
c***********************************************************************
c
      implicit double precision (a-h,o-z)
c
      dimension y(2), yp(2)
c
      external P
      external Q
c
      yp(1) = y(2)/P(x)
      yp(2) = Q(x)*y(1)
c
      return
      end
c
      function P(x)
c
c***********************************************************************
c     Function P of second order differential equation                 *
c***********************************************************************
c
      implicit double precision (a-h,o-z)
c
      common / comvar1 / theta, eimin, eimax, sigma, tau, aknul2
      common / comact / actoms
c
      cos2 = cos(theta)**2
      P = -(1.+sigma*x)*actoms + aknul2*cos2
c
      return
      end
c
      function Q(x)
c
c***********************************************************************
c     Function Q of second order differential equation                 *
c***********************************************************************
c
      implicit double precision (a-h,o-z)
c
      common / comvar1 / theta, eimin, eimax, sigma, tau, aknul2
      common / comact / actoms
c
      cos2 = cos(theta)**2
      Q = -aknul2*((1.+sigma*x)*actoms - aknul2*cos2 + sigma*tau)
c
      return
      end
c
      subroutine ploteif(ne,a)
c
c***********************************************************************
c     Plot eigenfunctions aksi(x,ne), eta(x,ne)                        *
c***********************************************************************
c
      implicit double precision (a-h,o-z)
c
      parameter (nmax = 100)
      parameter (imax = 1000)
      common / comvar2 / neiv, int, idiag
      common / comgrid / x(0:imax)
      common / eifun / aksi(0:imax,nmax), eta(0:imax,nmax)
c
      character*80 text
c
      anorm = 1.e-8
      do 10 i = 0,int
         if(abs(aksi(i,ne)).gt.anorm) anorm = abs(aksi(i,ne))
         if(abs(eta(i,ne)).gt.anorm)  anorm = abs(eta(i,ne))
   10 continue
      do 20 i = 0,int
         aksi(i,ne) = aksi(i,ne)/anorm
         eta(i,ne)  = eta(i,ne)/anorm
   20 continue
c
      write(text,'(''Eigenf. n ='',i2,'', oms ='',1pe12.4,0p)')
     &               ne, a
      mx = 2
      my = 2
      ipos = mod(ne,4) 
      if((ipos.eq.2).or.(ipos.eq.0)) mx = 3
      if((ipos.eq.3).or.(ipos.eq.0)) my = 3
      my = my + 100
      call nframe(mx,my,1,0.D+0,1.D+0,-1.D+0,1.D+0,
     &            text,32,'x',1,'ksi, eta',8)
      call lplot(mx,my,1,x(0),aksi(0,ne),-int-1,1,' ',0,' ',0,' ',0)
      call dplot(mx,my,x(0),eta(0,ne),int+1,1,90,90)
      write(6,'('' * plot eigenfunction'')')
c
      return
      end

